<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\Event;
use App\Models\EventProvince;
use App\Models\Province;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class adminEventProvinceController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index($event_id)
    {
        $event_info = Event::query()->where('id', $event_id)->firstOrFail();
        $events = EventProvince::query()->where('event_id', $event_id)->latest()->paginate(15);

        return view('admin.events.provinces.index', compact('events', 'event_info'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create($event_id)
    {
        $event_info = Event::query()->where('id', $event_id)->firstOrFail();
        $provinces = Province::query()->orderBy('name')->get();
        $events = Event::query()->latest()->get();

        return view('admin.events.provinces.create', compact('provinces', 'events', 'event_info'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $input = $request->all();

        $validation = Validator::make($input, [
            'top_img' => 'required|mimes:jpg,png,jpeg|max:2048', //2 MB
            'img' => 'required|mimes:jpg,png,jpeg|max:1024', //1 MB
            'province' => 'required|string|max:255',
            'description_video' => 'required|string|max:255',
            'description_video_poster' => 'required|string|max:255',
            'description' => 'required|string|max:5000',
            'description_title' => 'required|string|max:255',
            'event_id' => 'required',
        ]);

        if ($validation->fails()) {
            alert()->error($validation->errors()->first(), 'خطا !');
            return back()->withErrors($validation->errors())->withInput();
        }

        $file = $request->file('top_img');
        $file_ext = $file->getClientOriginalExtension();
        $file_name = 'event_top_' . time() . '.' . $file_ext;
        $top_img = $file->move('site/assets/events', $file_name);

        $file = $request->file('img');
        $file_ext = $file->getClientOriginalExtension();
        $file_name = 'event_' . time() . '.' . $file_ext;
        $img = $file->move('site/assets/events', $file_name);

        EventProvince::query()->create([
            'event_id' => $input['event_id'],
            'event_top_img' => $this->repair_file_src($top_img),
            'event_img' => $this->repair_file_src($img),
            'event_description_video' => $input['description_video'],
            'event_description_video_poster' => $input['description_video_poster'],
            'event_description' => $input['description'],
            'event_description_title' => $input['description_title'],
            'province_id' => $input['province'],
        ]);

        alert()->success('', 'رویداد با موفقیت افزوده شد');
        return redirect()->route('admin.event_provinces.index', ['event_id' => $input['event_id']]);
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $event_info = EventProvince::query()->findOrFail($id);
        $provinces = Province::query()->orderBy('name')->get();
        $events = Event::query()->latest()->get();

        return view('admin.events.provinces.edit', compact('provinces', 'event_info', 'events'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $input = $request->all();

        $event_info = EventProvince::query()->findOrFail($id);

        $validation = Validator::make($input, [
            'top_img' => 'nullable|mimes:jpg,png,jpeg|max:2048', //2 MB
            'img' => 'nullable|mimes:jpg,png,jpeg|max:1024', //1 MB
            'province' => 'required|string|max:255',
            'description_video' => 'required|string|max:255',
            'description_video_poster' => 'required|string|max:255',
            'description' => 'required|string|max:5000',
            'description_title' => 'required|string|max:255',
            'event_id' => 'required',
        ]);

        if ($validation->fails()) {
            alert()->error('', $validation->errors()->first());
            return back()->withErrors($validation->errors())->withInput();
        }

        if ($request->has('top_img')) {
            if (file_exists($event_info['event_top_img']) and !is_dir($event_info['event_top_img'])) {
                unlink($event_info['event_top_img']);
            }
            $file = $request->file('top_img');
            $file_ext = $file->getClientOriginalExtension();
            $file_name = 'event_top_' . time() . '.' . $file_ext;
            $top_img = $file->move('site/assets/events', $file_name);

            $event_info->update([
                'event_top_img' => $this->repair_file_src($top_img),
            ]);
        }

        if ($request->has('img')) {
            if (file_exists($event_info['event_img']) and !is_dir($event_info['event_img'])) {
                unlink($event_info['event_img']);
            }
            $file = $request->file('img');
            $file_ext = $file->getClientOriginalExtension();
            $file_name = 'event_' . time() . '.' . $file_ext;
            $img = $file->move('site/assets/events', $file_name);

            $event_info->update([
                'event_img' => $this->repair_file_src($img),
            ]);
        }

        $event_info->update([
            'event_id' => $input['event_id'],
            'event_description_video' => $input['description_video'],
            'event_description_video_poster' => $input['description_video_poster'],
            'event_description' => $input['description'],
            'event_description_title' => $input['description_title'],
            'province_id' => $input['province'],
        ]);

        alert()->success('', 'رویداد با موفقیت ویرایش شد');
        return back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    function repair_file_src($src)
    {
        return str_replace('\\', '/', $src);
    }
}
